﻿#include "precompiled.h"
#include "helpers.h"
#include "Texture.h"

#include "DDSTextureLoader.h"

namespace RTCam {

Texture::Texture(_In_z_ const char* name) :
	Resource(name),
	m_texture(nullptr),
	m_shaderView(nullptr),
	m_width(0),
	m_height(0),
	m_mipmapEnabled(false)
{
}

Texture::~Texture(void)
{
}

void Texture::ReleaseResources()
{
	m_texture = nullptr;
	m_shaderView = nullptr;
	m_width = 0;
	m_height = 0;
	m_mipmapEnabled = false;
}

void Texture::InitTexture(_In_ ID3D11Device1* device, size_t width, size_t height, DXGI_FORMAT texFormat, DXGI_FORMAT srvFormat, int bindFlags, bool mipmapEnabled)
{
	m_mipmapEnabled = mipmapEnabled;
	m_width = width;
	m_height = height;

	size_t mipLevels = 1;
	size_t miscFlags = 0;

	if(mipmapEnabled) {
		mipLevels = 0; // 0 = automatically generate max number of mip levels
		miscFlags = D3D11_RESOURCE_MISC_GENERATE_MIPS;
	}

	// Create the texture
	CD3D11_TEXTURE2D_DESC textureDesc(
		texFormat,
		width,
		height,
		1,								// Only one texture (not a texture array)
		mipLevels,						// Number of mip levels (0 = auto)
		bindFlags,
		D3D11_USAGE_DEFAULT,
		0,								// CPU access flags
		1,								// Sample count (No multisampling)
		0,								// Sample quality
		miscFlags
		);

	ThrowIfFailed(
		device->CreateTexture2D(
		&textureDesc,
		nullptr,
		&m_texture
		)
		);

	// Create the shader resource view
	CD3D11_SHADER_RESOURCE_VIEW_DESC shaderResourceDesc(
		D3D11_SRV_DIMENSION_TEXTURE2D,
		srvFormat,
		0,
		-1
		);
	ThrowIfFailed(
		device->CreateShaderResourceView(
		m_texture.Get(),
		&shaderResourceDesc,
		&m_shaderView
		)
		);

	SetDebugObjectName(m_texture.Get(), m_name + "Texture");
	SetDebugObjectName(m_shaderView.Get(), m_name + "ShaderView");
}

size_t Texture::GetWidth() const
{
	return m_width;
}

size_t Texture::GetHeight() const
{
	return m_height;
}

void Texture::InitFromDDS(_In_ ID3D11Device1* device, _In_z_ const char* filename)
{
	// Load the file
	ComPtr<ID3D11Resource> texture;
	ThrowIfFailed(DirectX::CreateDDSTextureFromFile(device, Utf8ToUtf16(filename).c_str(), &texture, &m_shaderView));
	ThrowIfFailed(texture.As(&m_texture));

	// Retrieve the texture description
	D3D11_TEXTURE2D_DESC desc = {0};
	m_texture->GetDesc(&desc);

	// Get width and height, and check whether the image was mip mapped
	m_width = desc.Width;
	m_height = desc.Height;
	m_mipmapEnabled = desc.MipLevels > 1;
}


} // end namespace